/*
 * Copyright (C) 2012 Edge-Core Networks
 * This software file (the "File") is owned and distributed by 
 * Edge-Core Networks under the following licensing terms.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */
/*-------------------------------------------------------------------------
 * Purpose: 
 *  This module Support to get partition information by index/type/name.
 *
 * Notes:
 *  2005, alvin ling, initial version.
 *  2008/06/30, jerry.du, rewrite for filemapping management.
 *  2010/02/17, aaron lien, Modify partition management with backup/restore functions.
 */

#include <common.h>
#include <asm/byteorder.h> /* for ntohl() */
#include <command.h>
#include <malloc.h>

#ifdef CONFIG_AMS_PARTITION

#include <ams_common.h>
#include <ams_part.h>

/*
 * you can change AMS_PART_LOG_LEVEL to 4 or smaller to enable debug msg in uc_mgr.c
 * you should use AMS_PRINTK("<%d>...", AMS_PART_LOG_LEVEL, ...) 
 * if you are adding debug msg 
 */
#define AMS_PART_LOG_LEVEL 5

#ifdef _USR_DEBUG // ,  debug print
#define AMS_DEBUG
#undef debug
#define debug(fmt,args...)	printf (fmt ,##args)
#endif	/* _USR_DEBUG */

/*
 * Local Variable Declare
 */


/* 
 * 2010/02/17, aaron lien , Modify partition management with backup/restore functions.
 * EPR: ES4524MA-HPoE-7LF-LN-00341, Backup tables in flash memory.
 */
/* The default value of partition table and H/W information are definded
 * in external module. Those values are the design requirement 
 * of each board.
 */
#define AMS_BPB 1

static int m_ams_flag=0;                   /* status of AMS */                                           
#define AMS_FLAG_INIT       0x01           /* ams_flag[0]: 1 = AMS is ready, 0 = not ready */
                                           
static AMS_BPB_T *m_bpb_default = NULL;    /* the default value of BPB */
static AMS_BPB_T *m_bpb_primary = NULL;    /* the primary BPB */
static AMS_BPB_T *m_bpb_backup = NULL;     /* the backup BPB */
static int m_bpb_backup_dirty;             /* 0 = BPB is not changed; 
                                            * 1 = BPB has been changed, it needs to copy BPB to flash device 
                                            */

#define BPB_FLAG_PT         0x0001      /* status bit of partition table */
#define BPB_FLAG_FMT        0x0002      /* status bit of file mapping table */
#define BPB_FLAG_HIT        0x0004      /* status bit of H/W info table */
#define BPB_FLAG_PRIMARY    0x0100      /* ID bit of primary area */
#define BPB_FLAG_BAK        0x0200      /* ID bit of backup area */
#define BPB_FLAG_MASK_DATA  0x00FF      /* mask of data bit */
#define BPB_MAX_RETRY_COUNT 3           /* the retry count for BPB recovery */


static int AMS_PART_init_BPB(AMS_BPB_T *bpb_ptr);
static int AMS_PART_set_bpb_checksum(AMS_BPB_T *bpb_ptr);
static int AMS_PART_write_BackupBPB(void);
static int AMS_PART_set_primary_part_table(AMS_PARTITION_TABLE_T *src);
static int AMS_PART_set_primary_filemap_table(AMS_FILEMAPPING_MGR_T *src);
static int AMS_PART_set_primary_hwinfo(FS_HW_Info_T *src);
static int AMS_PART_check_part_table(AMS_PARTITION_TABLE_T *part_tbl);
static int AMS_PART_check_filemap_table(AMS_FILEMAPPING_MGR_T *fmt);
static int AMS_PART_check_hwinfo_table(FS_HW_Info_T *hit);
static int AMS_PART_get_partbl_field_by_type(AMS_PARTITION_TABLE_T *part_tbl, AMS_PARTITION_TYPE_T type, FlashRegion_T *region);
static int AMS_PART_verify_primary(AMS_PARTITION_TABLE_T *part_tbl);
static int AMS_PART_verify_backup(AMS_BPB_T *pbp);
static int AMS_PART_recover_part_table(int flag_prim, int flag_bak);
static int AMS_PART_recover_filemap_table(int flag_prim, int flag_bak);
static int AMS_PART_recover_hwinfo(int flag_prim, int flag_bak);
static int do_BPB(cmd_tbl_t *cmdtp, int flag, int argc, char *argv[]);

#ifdef CONFIG_AMS_COMPACT_FLASH
extern unsigned long AMS_CFLIB_read_data(unsigned long sector,unsigned long sectors, unsigned char *buf);
extern unsigned long AMS_CFLIB_write_data(unsigned long sector,unsigned long sectors, unsigned char *buf, unsigned char format);
#endif

extern void memmovel(unsigned long dest, unsigned long src, size_t count);
extern int do_delete (cmd_tbl_t *cmdtp, int flag, int argc, char *argv[]);
extern int do_startup_file (cmd_tbl_t *cmdtp, int flag, int argc, char *argv[]);
extern unsigned long AMS_CMD_PART_set_startup_flag(unsigned long index, unsigned char flag);


static unsigned long
AMS_PART_check_type(unsigned long index, AMS_PARTITION_TYPE_T type)
{
    if (index > MAX_PARTITION_NUMBER)
    {
        AMS_PRINTK("<%d>%s:invalid partition index\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    if (m_bpb_primary->pat_table.flash_region[index].region_type == type)
        return RC_OK;

    return RC_ERROR;
}

static unsigned long
AMS_PART_check_free(unsigned long index)
{
    AMS_FILEMAPPING_MGR_T *fmt;
    
    AMS_PART_get_filemapping_mgr_ptr(&fmt);
    
    if (index > MAX_PARTITION_NUMBER)
    {
        AMS_PRINTK("<%d>%s:invalid partition index\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

#ifdef CONFIG_ACCTON_MULTI
    if (m_bpb_primary->pat_table.flash_region[index].region_type == TYPE_RUNTIME)
    {
        if (fmt->filemapping[index].startup_flag & AMS_PARTITION_FREE)
            return RC_OK;
        return RC_ERROR;
    }
#endif
#ifdef CONFIG_ACCTON_KERNEL_ROOTFS
    if (m_bpb_primary->pat_table.flash_region[index].region_type == TYPE_KERNEL  || 
	    m_bpb_primary->pat_table.flash_region[index].region_type == TYPE_ROOT_FS)
    {
	    if (fmt->filemapping[index].startup_flag & AMS_PARTITION_FREE)
		 	return RC_OK;
	    return RC_ERROR;
    }
#endif
	return RC_OK;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_check_active
 * ------------------------------------------------------------------------
 * PURPOSE  :   This funtion will check partition active mode.
 * INPUT    :   None
 * OUTPUT   :   None
 * RETURN   :   
 *              RC_OK  =  active mode
 *              RC_ERROR = inactive mode
 * NOTES    :   None
 * ------------------------------------------------------------------------
 */
unsigned long
AMS_PART_check_active(unsigned long index, AMS_PARTITION_TYPE_T type)
{
    AMS_FILEMAPPING_MGR_T *fmt;
    
    AMS_PART_get_filemapping_mgr_ptr(&fmt);
    
    if (index > MAX_PARTITION_NUMBER)
    {
        AMS_PRINTK("<%d>%s:invalid partition index\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    if (fmt->filemapping[index].startup_flag & AMS_PARTITION_ACTIVE)
	 	return RC_OK;

    return RC_ERROR;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_check_partition
 * ------------------------------------------------------------------------
 * PURPOSE  :   This funtion will check partition validity
 * INPUT      :   None
 * OUTPUT   :   None
 * RETURN    :   0/-1
 * NOTES     :   None
 * ------------------------------------------------------------------------
 */
unsigned long 
AMS_PART_check_partition(void)
{
#ifdef AMS_BPB
    return AMS_PART_check_part_table(&m_bpb_primary->pat_table);

#else
    if(pt_ptr->magic != PARTITION_TABLE_MAGIC)
        return RC_ERROR;

    if(pt_ptr->header_checksum != CheckSum((unsigned long)&pt_ptr->magic, 16))
        return RC_ERROR;

    return RC_OK;
#endif
}


/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_get_device_type
 * ------------------------------------------------------------------------
 * PURPOSE  :   This funtion will get device type
 * INPUT      :   device_name = the device name
 * OUTPUT   :   device_type = the device type
 * RETURN    :   -1 not found 0 success
 * NOTES     :   None
 * ------------------------------------------------------------------------
 */
unsigned long 
AMS_PART_get_device_type(const char *device_name, AMS_PARTITION_DEVICE_TYPE_T *device_type)
{
    if (device_name == NULL || device_type == NULL)
    {
        AMS_PRINTK("<%d>%s:invalid params\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }
	
    if (strcmp(device_name, DEVICE_NOR_NAME) == 0)
    {
        *device_type = TYPE_DEVICE_NOR;
        return RC_OK;
    }
	  
    if (strcmp(device_name, DEVICE_NAND_NAME) == 0)
    {
        *device_type = TYPE_DEVICE_NAND;
        return RC_OK;
    }
	  
    if (strcmp(device_name, DEVICE_CF_NAME) == 0)
    {
        *device_type = TYPE_DEVICE_CFCARD;
        return RC_OK;
    }

    if (strcmp(device_name, DEVICE_SPI_NAME) == 0)
    {
        *device_type = TYPE_DEVICE_SPI;
        return RC_OK;
    }

    return RC_ERROR;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_get_filemap_elememt
 * ------------------------------------------------------------------------
 * PURPOSE  :   This funtion will get filemapping info from filemapping table
 * INPUT    :   index = the index of partition
 * OUTPUT   :   filemapping_ptr = the address of filemapping info. 
 * RETURN    :   The execute of this function
 * NOTES     :   None
 * ------------------------------------------------------------------------
 */
unsigned long 
AMS_PART_get_filemap_elememt(unsigned long index, AMS_FILEMAPPING_T *filemapping_ptr)
{
    AMS_FILEMAPPING_MGR_T *fmt;
    
    if (filemapping_ptr == NULL)
    {
        AMS_PRINTK("<%d>%s:invalid params\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }
    
    if (AMS_PART_check_partition() != 0)
        return RC_PART_CHECK_ERROR;

    if (index < 0 || index >= MAX_PARTITION_NUMBER)
        return RC_PART_NUMBER_EXCEED;

    if (AMS_PART_get_filemapping_mgr_ptr(&fmt) != RC_OK)
        return RC_ERROR;
        
    *filemapping_ptr = fmt->filemapping[index];

    return RC_OK;
}


/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_get_filemapping_region
 * ------------------------------------------------------------------------
 * PURPOSE  :   Get partition data of the filemapping from partition table
 * INPUT    :   None
 * OUTPUT   :   region_ptr = partition data of filemapping. 
 * RETURN   :   The execute of this function
 * NOTES    :   None
 * ------------------------------------------------------------------------
 */

unsigned long AMS_PART_get_filemapping_region(FlashRegion_T *region_ptr)
{
#ifdef AMS_BPB
    return AMS_PART_get_partbl_field_by_type(&m_bpb_primary->pat_table, TYPE_FILEMAPPING_TABLE, region_ptr);

#else
    int index;
    FlashRegion_T region;

    if (region_ptr == NULL)
    {
        AMS_PRINTK("<%d>%s:invalid params\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    for (index = 0; index < MAX_PARTITION_NUMBER; index++)
    {
        memset(&region, 0, sizeof(FlashRegion_T));
        if (AMS_PART_get_region(index, &region) == 0)
            if (region.region_type == TYPE_FILEMAPPING_TABLE)
                break;
    }

    if (index == MAX_PARTITION_NUMBER)
        return RC_ERROR;

    //printf("AMS_PART_get_filemapping_region 0x%08x\n", region.region_base);
    *region_ptr = region;

    return RC_OK;
#endif    
}

unsigned long 
AMS_PART_init_filemapping(unsigned long index, AMS_FILEMAPPING_T *filemapping_ptr)
{
    unsigned long checksum;
    
    if (filemapping_ptr == NULL)
    {
        debug("%s: invalid params\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }
	
    filemapping_ptr->file_name[0] = '\0';
    filemapping_ptr->file_size = 0;
    filemapping_ptr->creat_time = 0;
    filemapping_ptr->part_index = index;
    filemapping_ptr->file_type = TYPE_NOT_USED;
    filemapping_ptr->part_type = PART_TYPE_NO_USED;
    filemapping_ptr->startup_flag = AMS_PARTITION_FREE;
    memset((char *)filemapping_ptr->reserved, 0, sizeof(unsigned long) * 40);

    AMS_PART_find_filemap_element_checksum(filemapping_ptr, &checksum);
    filemapping_ptr->part_checksum = checksum;
    
    return RC_OK;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_set_partition_checksum
 * ------------------------------------------------------------------------
 * PURPOSE :  set the checksum of the partition table.
 * INPUT   :  part_table - the partition table.
 * OUTPUT  :  None
 * RETURN  :  
 *              RC_OK = partition table is correct.
 *              <others> = invalid partition table.
 * NOTES   :  NONE
 * ------------------------------------------------------------------------
 */
static int AMS_PART_set_partition_checksum(AMS_PARTITION_TABLE_T *part_table)
{
    int i;
    
    if (part_table == NULL){
        debug("%s: invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    part_table->header_checksum = CheckSum((unsigned long)part_table, 16);
	
    for (i = 0; i < MAX_PARTITION_NUMBER; i++){
        part_table->flash_region[i].region_checksum = 
            CheckSum((unsigned long)(&part_table->flash_region[i].region_name), 
            sizeof(FlashRegion_T) - sizeof(unsigned long));
    }
        
    return RC_OK;
}


/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_find_filemap_element_checksum
 * ------------------------------------------------------------------------
 * PURPOSE :  find the checksum of file map element.
 * INPUT   :  filemap - file map element.
 * OUTPUT  :  chksum  - the checksum
 * RETURN  :  
 *              RC_OK = partition table is correct.
 *              <others> = invalid partition table.
 * NOTES   :  NONE
 * ------------------------------------------------------------------------
 */
int AMS_PART_find_filemap_element_checksum(AMS_FILEMAPPING_T *filemap, unsigned long *chksum)
{
    if ((!filemap) || (!chksum)) {
        debug("%s:invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }
	
    *chksum = CheckSum((unsigned long)&filemap->file_name[0], 
        sizeof(AMS_FILEMAPPING_T) - sizeof(unsigned long));

    return RC_OK;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_get_number
 * ------------------------------------------------------------------------
 * PURPOSE :   This funtion will the total number of the partition table
 * INPUT   :   None
 * OUTPUT  :   num - the number of the partition table
 * RETURN  :  
 *              RC_OK = partition table is correct.
 *              <others> = invalid partition table.
 * NOTES   :   NONE
 * ------------------------------------------------------------------------
 */
static unsigned long
AMS_PART_get_number(unsigned long *num)
{
    if (num == NULL){
        debug("%s:invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    *num = MAX_PARTITION_NUMBER;

    return RC_OK;
}

unsigned long
AMS_PART_get_partition_table_ptr(AMS_PARTITION_TABLE_T **ptr)
{
    *ptr= &m_bpb_primary->pat_table;
    return RC_OK;
}

unsigned long
AMS_PART_get_filemapping_mgr_ptr(AMS_FILEMAPPING_MGR_T **ptr)
{
    FlashRegion_T region;
    
    if (AMS_PART_get_filemapping_region(&region) != RC_OK)
        return RC_ERROR;
    *ptr= (AMS_FILEMAPPING_MGR_T *)region.region_base;

    return RC_OK;
}

int AMS_PART_get_default_part_table(AMS_PARTITION_TABLE_T *part_table)
{
    if (!part_table)
        return RC_ERROR;
    
    memcpy(part_table, &m_bpb_default->pat_table, sizeof(AMS_FILEMAPPING_MGR_T));

    return RC_OK;
}

int AMS_PART_get_default_filemapp_table(AMS_FILEMAPPING_MGR_T *filemap)
{  
    if (!filemap)
        return RC_ERROR;
    
    memcpy(filemap, &m_bpb_default->file_map, sizeof(AMS_FILEMAPPING_MGR_T));

    return RC_OK;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_get_hwinfo
 * ------------------------------------------------------------------------
 * PURPOSE  : Get the H/W information
 * INPUT    :   hit = memory buffer of H/W information
 * OUTPUT   :   the H/W information
 * RETURN   :   
 *              RC_OK = Success
 *              <others> = Fail
 * NOTES    :   NONE
 * ------------------------------------------------------------------------
 */
int AMS_PART_get_hwinfo(FS_HW_Info_T *hit)
{
    FlashRegion_T region;

    if(AMS_PART_get_partbl_field_by_type(&m_bpb_primary->pat_table, TYPE_HWINFO, &region) != RC_OK)
        return RC_ERROR;

    memcpy((unsigned char *)hit, (unsigned char *)region.region_base, sizeof(FS_HW_Info_T));
    
    return RC_OK;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_get_startup_by_type
 * ------------------------------------------------------------------------
 * PURPOSE  :   This funtion will get startup-flag partition index from filemapping table
 * INPUT      :   type = partition type
 * OUTPUT   :   partiton_idx = partition index
 * RETURN    :   The result of the function.
 * NOTES     :   NONE
 * ------------------------------------------------------------------------
 */
unsigned long 
AMS_PART_get_startup_by_type(AMS_PARTITION_TYPE_T type,  unsigned long *partiton_idx)
{
    int i;
    AMS_FILEMAPPING_MGR_T *fmt;
    
    if (partiton_idx == NULL){
        debug("%s:invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    AMS_PART_get_filemapping_mgr_ptr(&fmt);
    for (i = 0; i < MAX_PARTITION_NUMBER; i++)
    {	
        if ((m_bpb_primary->pat_table.flash_region[i].region_type == type) &&
            (fmt->filemapping[i].startup_flag & AMS_PARTITION_ACTIVE))
        {
            *partiton_idx = i;
            return RC_OK;
        }
    }

    return RC_ERROR;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_get_info_by_index
 * ------------------------------------------------------------------------
 * PURPOSE  :   This funtion will get partition address & size from partition table
 * INPUT      :   index = partition index
 * OUTPUT   :   addr = the address of the partition
                      size  = the size of partition 
 * RETURN    :   The result of the function.
 * NOTES     :   NONE
 * ------------------------------------------------------------------------
 */
unsigned long 
AMS_PART_get_info_by_index(unsigned long index, unsigned long *addr, unsigned long *size)
{
    FlashRegion_T region;

    if (addr == NULL || size == NULL){
        debug("%s:invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    if (AMS_PART_get_region(index, &region) == 0)
    {
        if ((region.region_type == TYPE_NOT_USED) || (region.region_type == TYPE_MAX))
            return RC_ERROR;
        *addr = region.region_base;
        *size = region.region_size;
        return RC_OK;
    }
    else
        return RC_ERROR;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_get_info_by_type
 * ------------------------------------------------------------------------
 * PURPOSE  :   This funtion will get partition address & size from partition table
 * INPUT      :   type = partition type
 * OUTPUT   :   addr = the address of the partition
                      size  = the size of partition 
 * RETURN    :   0/-1
 * NOTES     :   If there are more than one partition with same type, the function will 
 *                   search the active partition first, if there is no ative partition, it will get
 *                   the first paritition that match the given type.
 * ------------------------------------------------------------------------
 */
unsigned long 
AMS_PART_get_info_by_type(AMS_PARTITION_TYPE_T type, 
                          unsigned long *addr, 
                          unsigned long *size,
                          unsigned long part_index,
                          AMS_PARTITION_DEVICE_TYPE_T *device_type)
{
    unsigned char part_exist = FALSE;
    unsigned char part_free = FALSE;
    FlashRegion_T region;

    if (addr == NULL || size == NULL || device_type == NULL){
        debug("%s:invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }
    
    if (AMS_PART_check_type(part_index, type) == RC_OK)
    {
    	if (AMS_PART_check_free(part_index) == RC_OK &&  AMS_PART_check_active(part_index, type) == RC_ERROR)
    	{
            if (AMS_PART_get_region(part_index, &region) == RC_OK)
            {
                *addr = region.region_base;
                *size = region.region_size;
                if (AMS_PART_get_device_type((char *)region.device_name, device_type) != 0)
                {
                    AMS_PRINTK("<%d>%s:get device type error\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
                    return RC_ERROR;
                }
                return RC_OK;
            }
            part_free = TRUE;
        }
        part_exist = TRUE;
    }

    if (!part_exist)
    {
        printf("Partition does not exist\n");
        return RC_ERROR;
    }

    if (!part_free)
    {
        if(AMS_PART_check_active(part_index, type) == RC_OK){
            printf("This partition is active mode.\n");            
        }else{
            printf("There is no free region.\n");
        }
        return RC_ERROR;
    }

    return RC_OK;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_set_part_info_by_index
 * ------------------------------------------------------------------------
 * PURPOSE  :   This funtion will set partition to active , name and file size.
 * INPUT    :   part_index = partition index
 *              name = name of file to be saved in partition.
 *              file_size = file size.
 * OUTPUT   :   None
 * RETURN   :   
 *              RC_OK = Success,
 *              RC_PARAMENT_ERROR = error on input value.
 * NOTES    :   NONE
 * ------------------------------------------------------------------------
 */
unsigned long AMS_PART_set_part_info_by_index(unsigned long part_index, const char *name, 
                                unsigned long file_size,
                                AMS_PARTITION_TYPE_T type)
{
    AMS_FILEMAPPING_MGR_T current_filemap_table;
    AMS_FILEMAPPING_MGR_T *fmt;

    if (name == NULL){
        debug("%s:invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

	if (part_index < 0 || part_index > MAX_PARTITION_NUMBER)
    {
        AMS_PRINTK("<%d>%s:invalid partition index\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }
	
	AMS_PART_get_filemapping_mgr_ptr(&fmt);
	current_filemap_table = *fmt;

    strcpy((char *)current_filemap_table.filemapping[part_index].file_name, (char *)name);
	current_filemap_table.filemapping[part_index].file_name[PART_FILE_NAMELEN-1] = '\0';
    current_filemap_table.filemapping[part_index].file_size = file_size;
    current_filemap_table.filemapping[part_index].creat_time = 0;
    current_filemap_table.filemapping[part_index].part_index = part_index;
    current_filemap_table.filemapping[part_index].file_type = type;
    current_filemap_table.filemapping[part_index].part_type = PART_TYPE_FIXED;
	current_filemap_table.filemapping[part_index].startup_flag &= ~AMS_PARTITION_FREE;
    current_filemap_table.filemapping[part_index].startup_flag |= AMS_PARTITION_INACTIVE;
	current_filemap_table.filemapping[part_index].part_checksum = 
       CheckSum((unsigned long)&(current_filemap_table.filemapping[part_index].file_name), 
         sizeof(AMS_FILEMAPPING_T) - sizeof(unsigned long));
    
	AMS_PART_write_filemapping2flash(&current_filemap_table);

    return RC_OK;
}

unsigned long 
AMS_PART_get_info_by_name(unsigned char *name, 
                          unsigned long *addr, 
                          unsigned long *size,
                          AMS_PARTITION_TYPE_T *type)
{
    unsigned long i;
    FlashRegion_T region;
    unsigned long part_num;

    if (name == NULL || addr == NULL || size == NULL || type == NULL){
        debug("%s:invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }
    
    AMS_PART_get_number(&part_num);
    for (i = 0; i < part_num; i++)
    {
        if (AMS_PART_get_region(i, &region) == 0)
        { 
            if (strcmp((char *)region.region_name, (char *)name) == 0)
            {
                *addr = region.region_base;
                *size = region.region_size;
                *type = region.region_type;
                return RC_OK;
            }
        }
    }
  
    return RC_ERROR;
}

unsigned long 
AMS_PART_get_info_by_filename(const char *filename, 
                              unsigned long *addr, 
                              unsigned long *size,
                              unsigned long *part_idx,
                              AMS_PARTITION_DEVICE_TYPE_T *device_type)
{
    unsigned long i;
    FlashRegion_T region;
    AMS_FILEMAPPING_T filemapping;
    unsigned long part_num;

    if (filename == NULL || addr == NULL || size == NULL 
        || part_idx == NULL || device_type == NULL)
    {
        debug("%s:invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }
    
    AMS_PART_get_number(&part_num);
    for (i = 0; i < part_num; i++)
    {
        if (AMS_PART_get_filemap_elememt(i, &filemapping) == 0)
        {
            if (strcmp((char *)filemapping.file_name, (char *)filename) == 0)
            {
                if (AMS_PART_get_region(i, &region) == 0)
                {
                    *addr = region.region_base;
                    *size = region.region_size;
                    *part_idx = i;
                    if (AMS_PART_get_device_type((char *)region.device_name, device_type) != 0)
                        return RC_ERROR;
                    return RC_OK;
                }
            }
        }
    }
    return RC_ERROR;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_get_partition_filename
 * ------------------------------------------------------------------------
 * PURPOSE  :   This funtion will get partition name by index
 * INPUT      :   index = the index of the partition
 * OUTPUT   :   part_name - the name associated the partition
 * RETURN    :   -1/0
 * NOTES     :   None
 * ------------------------------------------------------------------------
 */	
unsigned long 
AMS_PART_get_partition_filename(unsigned long index, 
    unsigned char *file_name, unsigned long *file_length)
{
    AMS_FILEMAPPING_MGR_T *fmt;
    
    if (file_name == NULL || file_length == NULL){
        debug("%s:invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }
    if (index < 0 || index > MAX_PARTITION_NUMBER){
        debug("%s:invalid partition index\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    AMS_PART_get_filemapping_mgr_ptr(&fmt);
    if(fmt->filemapping[index].part_checksum != 
        CheckSum((unsigned long)&(fmt->filemapping[index].file_name), 
        sizeof(AMS_FILEMAPPING_T) - sizeof(unsigned long)))
    {
        return RC_ERROR;
    }
    strcpy((char *)file_name, (char *)fmt->filemapping[index].file_name);
    *file_length = fmt->filemapping[index].file_size;

    return RC_OK;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_get_region
 * ------------------------------------------------------------------------
 * PURPOSE  :   This funtion will get partition region from partition table
 * INPUT      :   index = the index of partition
 * OUTPUT   :   region = the region of the given partition index. 
 * RETURN    :   The execute of this function
 * NOTES     :   None
 * ------------------------------------------------------------------------
 */
unsigned long 
AMS_PART_get_region(unsigned long index, FlashRegion_T *region)
{
    if (region == NULL){
        debug("%s:invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    if (index < 0 || index >= MAX_PARTITION_NUMBER)
        return RC_PART_NUMBER_EXCEED;

    if (AMS_PART_check_partition() != 0)
        return RC_PART_CHECK_ERROR;

    if (m_bpb_primary->pat_table.flash_region[index].region_type == TYPE_NOT_USED)
        return RC_PART_DO_NOT_EXIST;

    if (m_bpb_primary->pat_table.flash_region[index].region_checksum != 
           CheckSum((unsigned long)&m_bpb_primary->pat_table.flash_region[index].region_name, 
                sizeof(FlashRegion_T) - sizeof(unsigned long)))
        return RC_PART_DO_NOT_EXIST;

    *region = m_bpb_primary->pat_table.flash_region[index];

    return RC_OK;
}


unsigned long
AMS_PART_get_boot_part(AMS_PARTITION_TYPE_T type, unsigned long *active_part_idx)
{
    int i;
	unsigned long part_num = 0;
	AMS_FILEMAPPING_T filemapping;
	AMS_FILEMAPPING_MGR_T *fmt;

    if (active_part_idx == NULL){
        debug("%s:invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

#ifndef ACCTON_PATCH    
	if (type < TYPE_NOT_USED || type > TYPE_MAX)
	{
		AMS_PRINTK("<%d>%s:invalid part type\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
		return RC_ERROR;
	}
#endif

    AMS_PART_get_filemapping_mgr_ptr(&fmt);
	AMS_PART_get_number(&part_num);
	for (i = 0; i < part_num; i++)
	{
       	if (m_bpb_primary->pat_table.flash_region[i].region_type != type)
            continue;
        
       	if (fmt->filemapping[i].startup_flag & AMS_PARTITION_ACTIVE)
       	{
       		*active_part_idx = i;
			break;
       	}
	}

	if (AMS_PART_get_filemap_elememt(*active_part_idx, &filemapping) !=  0)
	{
        AMS_PRINTK("<%d>%s:get part error by index\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
		return RC_ERROR;
	}

	if (filemapping.startup_flag & AMS_PARTITION_FREE)
	{
		AMS_PRINTK("<%d>%s:boot part is free\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
		return RC_ERROR;
	}
	
	if (!(filemapping.startup_flag & AMS_PARTITION_ACTIVE))
	{
		AMS_PRINTK("<%d>%s:boot part is not active\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
		return RC_ERROR;
	}
	return RC_OK;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_get_inactive_part
 * ------------------------------------------------------------------------
 * PURPOSE  :   Get index of the inactive partition with specific type.
 * INPUT    :   type = partition type
 * OUTPUT   :   part_idx = index of the partition. 
 * RETURN   :   
 *              RC_OK = Success
 *              <others> = Fail
 * NOTES    :   None
 * ------------------------------------------------------------------------
 */
int AMS_PART_get_inactive_part(AMS_PARTITION_TYPE_T type, unsigned long *part_idx)
{
    int i;
    AMS_FILEMAPPING_MGR_T *fmt;

    if (part_idx == NULL){
        debug("%s:invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    AMS_PART_get_filemapping_mgr_ptr(&fmt);
    
	for (i = 0; i < MAX_PARTITION_NUMBER; i++){
       	if (m_bpb_primary->pat_table.flash_region[i].region_type != type)
            continue;

        if (fmt->filemapping[i].startup_flag & AMS_PARTITION_FREE)
            continue;
        
       	if (fmt->filemapping[i].startup_flag & AMS_PARTITION_INACTIVE){
       		*part_idx = i;
			return RC_OK;
       	}
	}

	return RC_ERROR;
}

/* Don't do real protect for all sectors.
 * SPI Flash write protection on per sector is not supported,
 * and some partitions need to be read/write in runtime.
 */    
static unsigned long
AMS_PART_write_flash(unsigned long dest_addr, unsigned char *src, 
     unsigned long erase_length, unsigned long write_length, unsigned char protect)
{
    unsigned long err_code, phys_addr;
    
    debug("%s(): [dest_addr] 0x%X [src] 0x%X [erase_length] %ld [write_length] %ld [protect] %d\n",
          __FUNCTION__, dest_addr, src, erase_length, write_length, protect);
   
 
    if (src == NULL){
        debug("%s:invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    if(write_length > erase_length){
        debug("%s: Source data length(%ld) is over flash block size(%ld)\n", write_length, erase_length);
        return RC_ERROR;
    }
    
    /* map virtual flash address to physical address and do flash chip select */ 
    AMS_FLASH_MAPPING(dest_addr, phys_addr);

    udelay(10000);
    err_code = RC_OK;    
    /* flag protect means that the partition is protected, we should protect off it when write to it */
    if (protect > 0 )
    {
        if (flash_sect_protect(0, phys_addr, phys_addr + erase_length - 1))
        {
            AMS_PRINTK("<%d>%s:un-protech flash error\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
            err_code = RC_ERROR;
        }
    }

    udelay(10000);
    if (flash_sect_erase(phys_addr, phys_addr + erase_length - 1))
    {
        AMS_PRINTK("<%d>%s:erase flash error\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        err_code = RC_ERROR;
    }

    udelay(100000);
    if (flash_write((char *)src, phys_addr, write_length))
    {
        AMS_PRINTK("<%d>%s:write flash error\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        err_code = RC_ERROR;
    }

    if (protect == 1)
    {
        udelay(100000);
        if (flash_sect_protect(1, phys_addr, phys_addr + erase_length - 1)){
            AMS_PRINTK("<%d>%s:protech flash error\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
            err_code = RC_ERROR;
        }
    }
    
    AMS_FLASH_DEFAULT();    /* select on first flash as normal operation */ 

    return err_code;
}

unsigned long AMS_PART_write_part2flash(AMS_PARTITION_TABLE_T *partition_table_ptr)
{
    unsigned char p = 1;  // do real protect

    if (partition_table_ptr == NULL){
        debug("%s:invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }
    
#if 0   //2010/02/25, aaron lien, removed, due to EPR: ES4524MA-HPoE-7LF-LN-00343
    AMS_PARTITION_DEVICE_TYPE_T device_type;
    AMS_FILEMAPPING_MGR_T *fmt;
    FlashRegion_T   region;
    unsigned long region_mode;    

    /* get flash device type */
    device_type = TYPE_DEVICE_SPI;  /* default flash type */
    if(AMS_PART_get_region(PARTITION_MANAGEMENT_INDEX, &region) == RC_OK){
        AMS_PART_get_device_type((char *)region.device_name, &device_type);
    }
    debug("[Debug] %s: device type = %d\n", __FUNCTION__, device_type);

    /* Don't do real protect for all sectors.
     * SPI Flash write protection on per sector is not supported,
     * and some partitions need to be read/write in runtime.
     */
    region_mode=AMS_PART_get_region_mode(&region);
    if ((region_mode == PART_MODE_RW) || (device_type == TYPE_DEVICE_SPI))
        p = 2;  //Don't do real protect
#endif    

    AMS_PART_write_flash(PARTITION_MANAGEMENT_BASE_ADDR, (unsigned char *)partition_table_ptr, 
        CFG_FLASH_SECTOR_SIZE, sizeof(AMS_PARTITION_TABLE_T), p);

    ///* when the partition table is changed, we get new filemapping region address */
    //AMS_PART_get_filemapping_mgr_ptr(&fmt);


    {/* EPR: ES4524MA-HPoE-7LF-LN-00341 Backup tables in flash memory. */
        memcpy(&m_bpb_primary->pat_table, partition_table_ptr, sizeof(AMS_PARTITION_TABLE_T));        
        memcpy(&m_bpb_backup->pat_table, partition_table_ptr, sizeof(AMS_PARTITION_TABLE_T));
        AMS_PART_write_BackupBPB();
    }

    return RC_OK;
}

unsigned long 
AMS_PART_write_filemapping2flash(AMS_FILEMAPPING_MGR_T *filemapping_mgr)
{
    AMS_PARTITION_DEVICE_TYPE_T device_type;
    FlashRegion_T region;
    unsigned long rc;
    unsigned char p = 1;  // do real protect

    if (filemapping_mgr == NULL){
        debug("%s:invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    if (AMS_PART_get_filemapping_region(&region) != RC_OK)
        return RC_ERROR;
    
    AMS_PART_get_device_type((char *)region.device_name, &device_type);
    debug("device type = %d\n", device_type);

#if 0   //2010/02/25, aaron lien, removed, due to EPR: ES4524MA-HPoE-7LF-LN-00343
        /* Don't do real protect for all sectors.
         * SPI Flash write protection on per sector is not supported,
         * and some partitions need to be read/write in runtime.
         */    
    region_mode=AMS_PART_get_region_mode(&region);
    if ((region_mode == PART_MODE_RW) || (device_type == TYPE_DEVICE_SPI))
            p = 2;  //Don't do real protect
#endif

    rc = AMS_PART_write_flash(region.region_base, (unsigned char *)filemapping_mgr, 
            CFG_FLASH_SECTOR_SIZE, sizeof(AMS_FILEMAPPING_MGR_T), p);


    {/* EPR: ES4524MA-HPoE-7LF-LN-00341 Backup tables in flash memory. */
        memcpy(&m_bpb_primary->file_map, filemapping_mgr, sizeof(AMS_FILEMAPPING_MGR_T));        
        memcpy(&m_bpb_backup->file_map, filemapping_mgr, sizeof(AMS_FILEMAPPING_MGR_T));
        AMS_PART_write_BackupBPB();        
    }

    return rc;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_write_hwinfo
 * ------------------------------------------------------------------------
 * PURPOSE  : Copy the H/W information to flash device.
 * INPUT    :   hit = memory buffer of H/W information
 * OUTPUT   :   None.
 * RETURN   :   
 *              RC_OK = Success
 *              <others> = Fail
 * NOTES    :   NONE
 * ------------------------------------------------------------------------
 */
int AMS_PART_write_hwinfo(FS_HW_Info_T *hit)
{
    //if(AMS_PART_write_to_device(TYPE_HWINFO, PARTITION_HWINFO_FILENAME, (unsigned long)hit, sizeof(FS_HW_Info_T), PARTITION_HWINFO_INDEX) != RC_OK){
    //    printf("[Error] %s: Fail to save hardware information to flash\n", __FUNCTION__);
    //    return RC_ERROR;
    //}

    if (hit == NULL){
        debug("%s:invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    AMS_PART_set_primary_hwinfo(hit);
    

    {/* EPR: ES4524MA-HPoE-7LF-LN-00341 Backup tables in flash memory. */
        memcpy(&m_bpb_backup->hw_info, hit, sizeof(FS_HW_Info_T));
        AMS_PART_write_BackupBPB();        
    }
    
    return RC_OK;
}
/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_set_hwinfo_checksum
 * ------------------------------------------------------------------------
 * PURPOSE :  set the checksum of the partition table.
 * INPUT   :  part_table - the partition table.
 * OUTPUT  :  None
 * RETURN  :  
 *              RC_OK = partition table is correct.
 *              <others> = invalid partition table.
 * NOTES   :  NONE
 * ------------------------------------------------------------------------
 */
static int AMS_PART_set_hwinfo_checksum(FS_HW_Info_T *hit)
{
    if (hit == NULL){
        debug("%s: invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    CheckSum((unsigned long)hit, sizeof(FS_HW_Info_T) - sizeof(unsigned short));
	        
    return RC_OK;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_reset
 * ------------------------------------------------------------------------
 * PURPOSE  :   Set the current partition and file map to default value.
 * INPUT    :   None.
 * OUTPUT   :   None.
 * RETURN   :   
 *              RC_OK = Success
 *              <others> = Fail
 * NOTES    :   NONE
 * ------------------------------------------------------------------------
 */
int AMS_PART_reset(void)
{
    AMS_PART_set_primary_part_table(&m_bpb_default->pat_table);
    memcpy(&m_bpb_backup->pat_table, &m_bpb_default->pat_table, sizeof(AMS_PARTITION_TABLE_T));

    AMS_PART_set_primary_filemap_table(&m_bpb_default->file_map);
    memcpy(&m_bpb_backup->file_map, &m_bpb_default->file_map, sizeof(AMS_FILEMAPPING_MGR_T));

    return AMS_PART_write_BackupBPB();
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_reset_filemap
 * ------------------------------------------------------------------------
 * PURPOSE  :   Set the current file map to default value.
 * INPUT    :   None.
 * OUTPUT   :   None.
 * RETURN   :   
 *              RC_OK = Success
 *              <others> = Fail
 * NOTES    :   NONE
 * ------------------------------------------------------------------------
 */
int AMS_PART_reset_filemap(void)
{

    AMS_PART_set_primary_filemap_table(&m_bpb_default->file_map);
    memcpy(&m_bpb_backup->file_map, &m_bpb_default->file_map, sizeof(AMS_FILEMAPPING_MGR_T));

    return AMS_PART_write_BackupBPB();
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_reset_hwinfo
 * ------------------------------------------------------------------------
 * PURPOSE  :   Set the current H/W info to default value.
 * INPUT    :   None.
 * OUTPUT   :   None.
 * RETURN   :   
 *              RC_OK = Success
 *              <others> = Fail
 * NOTES    :   NONE
 * ------------------------------------------------------------------------
 */
int AMS_PART_reset_hwinfo(void)
{
    return AMS_PART_write_hwinfo(&m_bpb_default->hw_info);
}

unsigned long AMS_PART_write_to_device(AMS_PARTITION_TYPE_T type,
                                  const char *name,
                                  unsigned long addr, 
                                  unsigned long length,
                                  unsigned long part_index)
{
    AMS_PARTITION_DEVICE_TYPE_T device_type;
    unsigned long start_addr = 0, size = 0;    
    unsigned long ret = RC_ERROR;
     unsigned char p = 1;  // do real protect

    debug("\n%s(): [type] %d [name] %s [addr] 0x%X [length] %ld\n",__FUNCTION__, type, name, addr, length);

    if (type <= TYPE_NOT_USED || type >= TYPE_MAX){
        printf("[AMS] Invalid type of partation. type = %d\n", type);
		return RC_ERROR;		
    }

    if (AMS_PART_get_info_by_type(type, &start_addr, &size, part_index, &device_type) != RC_OK){
	   	printf("[AMS] Fail to find partition. type = %d, index = %ld\n", type, part_index);
		return RC_ERROR;
        
    }else{            
        debug("flash partation info:  start_addr=0x%X, size=%ld, device_type=%d\n", start_addr, size, device_type);

        
        switch (device_type){
            case TYPE_DEVICE_NOR:
                ret = AMS_PART_write_flash(start_addr, (unsigned char *)addr, size, length, p);
                break;
                
            case TYPE_DEVICE_SPI: 
                ret = AMS_PART_write_flash(start_addr, (unsigned char *)addr, size, length, p);
                break;

            #if 0   //def CONFIG_COMPACT_FLASH
            case TYPE_DEVICE_CFCARD:
                if (length > size){
                    AMS_PRINTK("<%d>%s:too large length\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
                    ret = RC_ERROR;
                    goto END;
                }
			   
                /* compact flash's write/read operation should be sector sized */
                start_addr /= CF_STANDARD_SECTOR_SIZE;
                /* 
                 * the format operation will cost a lot of time, write data to cf card directly
                 */
                size = length / CF_STANDARD_SECTOR_SIZE + 1;
                if (AMS_CFLIB_write_data(start_addr, size, (unsigned char*)(addr), FALSE)!= 0){
                    AMS_PRINTK("<%d>%s:write cfcard error\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
                    ret = RC_ERROR;
                    goto END;
                }
                break;
            #endif
            
            default:
                printf("[AMS] Unknow device type. device_type = %d\n", __FUNCTION__, device_type);
        }        
    }

    if(ret == RC_OK){
        debug("\nset partition info: \n");
        if (AMS_PART_set_part_info_by_index(part_index, name, length, type) != RC_OK){
            printf("[AMS] Fail to set partition info.\n");
            ret = RC_ERROR;
        }
    }
    
    return ret;
}

unsigned long
AMS_PART_read_from_device(AMS_PARTITION_TYPE_T type,
                                     const char *name,
                                     unsigned long length, 
                                     unsigned long *buf_addr)
{
    AMS_PARTITION_DEVICE_TYPE_T device_type;
    unsigned long start_addr = 0, phys_addr;
    unsigned long size = 0;
    unsigned long partition_index = 0;

    if (type <  TYPE_NOT_USED || type >= TYPE_MAX){
        debug("%s: invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    debug("%s: [type] %d, [name] %s, [length] %ld, [*buf_addr] 0x%08x\n", __FUNCTION__, type, name, length, *buf_addr);
    
    if (AMS_PART_get_info_by_filename((char *)name, &start_addr, &size, &partition_index,&device_type) == 0)
    {       
        debug("get flash data from: [start_addr]0x%lX, [size]%ld, [partition_index]%ld, [device_type]%d\n", start_addr, size, partition_index, device_type);   
        
        switch (device_type){
            case TYPE_DEVICE_NOR:
                *buf_addr = start_addr;
                break;

            case TYPE_DEVICE_SPI:
                if(!*buf_addr){
                    printf("[AMS] Invalid buffer address ! Null pointer !\n");
                    return RC_ERROR;
                }

                /* map flash address to physical address and do flash chip select */ 
                AMS_FLASH_MAPPING(start_addr, phys_addr);                
                //memmove ((char *)*buf_addr, (char *)start_addr, length);  
                memmovel (*buf_addr, phys_addr, length); //EPR: ES3526MA-HPoE-7LF-LN-00016, Speed booting process.
                AMS_FLASH_DEFAULT();    /* select on first flash as normal operation */                 
                break;

            case TYPE_DEVICE_NAND:
                break;

#ifdef CONFIG_COMPACT_FLASH
            case TYPE_DEVICE_CFCARD:
                start_addr /= CF_STANDARD_SECTOR_SIZE;
                length = length / CF_STANDARD_SECTOR_SIZE + 1;
                AMS_CFLIB_read_data(start_addr, length, (unsigned char *)*buf_addr);
                break;
#endif
            default:
               AMS_PRINTK("<%d>%s:invalid device type\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
               break;
        }
    }
    else
    {
        AMS_PRINTK("<%d>%s:failed to find linux part\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_ERROR;
    }
    return RC_OK;
}

/* ------------------------------------------------------------------------
 * FUNCTION NAME: AMS_PART_get_non_startup_filename
 * PURPOSE  : get the inactive file of a given type.
 * INPUT    : file_type   -- the type of file
 * OUTPUT   : 
 *            file_name   -- the file name 
 *            file_length -- the size of file.
 * RETURN   : 
 *            RC_OK    = Success
 *            RC_ERROR = Fail
 * NOTES    : None
 * ------------------------------------------------------------------------
 */
int AMS_PART_get_non_startup_filename(AMS_PARTITION_TYPE_T file_type, unsigned char *file_name, unsigned long *file_length)
{
    unsigned long part_idx;
    int result = RC_ERROR;

    if (AMS_PART_get_inactive_part(file_type, &part_idx) == RC_OK)
    {
        if (AMS_PART_get_partition_filename(part_idx, file_name, file_length) == RC_OK)
        {
            result= RC_OK;
        }
        else
        {
            debug("%s(): get part filename error ! type = %d, part_index = %d\n", __FUNCTION__, file_type, part_idx);
        }

    }
    else
    {
        debug("%s(): get inactive part error !  type =%d\n", __FUNCTION__, file_type);
    }

    return result;
}

/* ------------------------------------------------------------------------
 * FUNCTION NAME: AMS_PART_get_startup_filename
 * PURPOSE  : get the startup file of a given type.
 * INPUT    : 
 *            file_type   -- the type of file
 * OUTPUT   :
 *            file_name   -- the file name 
 *            file_length -- the size of file.
 * RETURN   :
 *            RC_OK    = Success
 *            RC_ERROR = Fail
 * NOTES    : None
 * ------------------------------------------------------------------------
 */
int AMS_PART_get_startup_filename(AMS_PARTITION_TYPE_T file_type, unsigned char *file_name, unsigned long *file_length)
{
    unsigned long part_idx = 0;
    int result = RC_ERROR;

    if (AMS_PART_get_boot_part(file_type, &part_idx) == RC_OK)
    {
        if (AMS_PART_get_partition_filename(part_idx, file_name, file_length) == RC_OK)
        {
            result= RC_OK;
        }
        else
        {
            debug("%s(): get part filename error ! type = %d, part_index = %d\n", __FUNCTION__, file_type, part_idx);
        }
    }
    else
    {
        debug("%s(): get boot part error !  type =%d\n", __FUNCTION__, file_type);
    }

    return result;
}

/* ------------------------------------------------------------------------
 * FUNCTION NAME: AMS_PART_recover_runtime_to_default
 * PURPOSE  :   Recover runtime file information and
 *              write the default file name, size.
 *              Bootable runtime image will be loaded at LA_DB_ADDR if it is
 *              recovered successfully.
 * INPUT    :   None
 * OUTPUT   :   None
 * RETURN   :   
 *              RC_OK    = Success
 *              RC_ERROR = Fail
 *
 * NOTES    :   This function shall check the runtime file in flash first.
 *              If runtime's checksum is correct, then using default file name.
 *              If there is environment "runtimename", get file name from "runtimename".
 *              Otherwise, runtime file name is the partition name with extension ".bix".
 *
 *              If runtime's checksum is not correct, then check next runtime file.
 * ------------------------------------------------------------------------
 */
int AMS_PART_recover_runtime_to_default(void)
{
    int i, result=RC_ERROR;
    image_header_t  header;
    char file_name[PART_FILE_NAMELEN];
    char *env_runtimefile_name;
    unsigned long checksum, file_size, runtime_base_addr, phys_addr;
    AMS_PARTITION_TABLE_T *pt_ptr;

    if(AMS_PART_get_partition_table_ptr(&pt_ptr)!=RC_OK)
    {
        debug("%s:AMS_PART_get_partition_table_ptr return error.\r\n");
        return result;
    }

    for (i = 0; i < MAX_PARTITION_NUMBER; i++)
    {
        debug("%s:region %d: %d\n", __FUNCTION__, i, pt_ptr->flash_region[i].region_type);

        if (pt_ptr->flash_region[i].region_type != TYPE_RUNTIME)
            continue;

        debug("%s:Runtime partition index = %d\n", __FUNCTION__,  i);

        runtime_base_addr=pt_ptr->flash_region[i].region_base;
        AMS_FLASH_MAPPING(runtime_base_addr, phys_addr);
        udelay(10000);
        memmove(&header, (char *)runtime_base_addr, sizeof(image_header_t));
        AMS_FLASH_DEFAULT();

        /* vaildate magic
         */
        if (header.ih_magic != IH_MAGIC)
        {
            debug("%s:Bad runtime file(image header magic byte error) on part idx %d(read val=%lu)!\n", __FUNCTION__, i, header.ih_magic);
            continue;
        }

        /* validate image header crc
         */
        checksum = ntohl(header.ih_hcrc);
        header.ih_hcrc = 0;
        if(crc32(0, (uchar*)&header, sizeof(image_header_t)) != checksum)
        {
            debug("%s:Bad image header checksum\n", __FUNCTION__);
            continue;
        }

        /* Runtime file name
         */
        env_runtimefile_name = getenv("runtimename");
        if(env_runtimefile_name)
            strcpy(file_name, env_runtimefile_name);
        else
            sprintf(file_name, "%s.bix", pt_ptr->flash_region[i].region_name); 

        file_size=sizeof(image_header_t)+ntohl(header.ih_size);
        debug("%s:Set to default: [part idx] %d [file name] %s, [file size] %ld\n", __FUNCTION__, i, file_name, file_size);

        if (AMS_PART_set_part_info_by_index(i, file_name, file_size, TYPE_RUNTIME) == RC_OK)
        {
            unsigned long boot_addr = LA_DB_ADDR;

            if(AMS_PART_read_from_device(TYPE_RUNTIME, file_name, file_size, &boot_addr)==RC_OK)
            {
                /* validate image data crc
                 */
                if(crc32(0, (uchar*)boot_addr + sizeof(image_header_t), file_size - sizeof(image_header_t))!=ntohl(header.ih_dcrc))
                {
                    /* delete invalid runtime file
                     */
                    debug("%s:Bad image data checksum. Delete file '%s'\n", __FUNCTION__, file_name);
                    AMS_PART_delete_runtime_file(file_name);
                    continue;
                }
            }
            else
            {
                debug("%s:Read part index %d error. Delete file '%s'\n", __FUNCTION__, i, file_name);
                /* delete invalid runtime file
                 */
                AMS_PART_delete_runtime_file(file_name);
            }

            /* recover successfully
             */
            result=RC_OK;
            break;
        }
        else
        {
            debug("%s: Fail to set part idx %d as default\n", __FUNCTION__, i);
            continue;
        }

    }

    return result;
}

/* ------------------------------------------------------------------------
 * FUNCTION NAME: AMS_PART_validate_and_load_boot_runtime
 * PURPOSE  :   Validate the startup runtime image(the runtime image set as active)
 *              The runtime image will be loaded to LA_DB_ADDR after returning
 *              from this function if there is no error.
 * INPUT    :   None
 * OUTPUT   :   None
 * RETURN   :   
 *              RC_OK    = Success
 *              RC_ERROR = Fail
 *
 * NOTES    : 1.This function will validate the startup runtime image through
 *              image header checksum and image data checksum. The runtime
 *              image will be loaded to LA_DB_ADDR if there is no error.
 *            2.Invalid runtime image will be deleted from file mapping table.
 * ------------------------------------------------------------------------
 */
int AMS_PART_validate_and_load_boot_runtime(void)
{
    unsigned long active_part_idx, checksum, boot_addr = LA_DB_ADDR;
    unsigned long file_size;
    image_header_t  header;
    char file_name[PART_FILE_NAMELEN];

    if(AMS_PART_get_boot_part(TYPE_RUNTIME, &active_part_idx)!=RC_OK)
        return RC_ERROR;

    if(AMS_PART_get_partition_filename(active_part_idx, (unsigned char*)file_name, &file_size)!=RC_OK)
        return RC_ERROR;

    if(AMS_PART_read_from_device(TYPE_RUNTIME, file_name, file_size, &boot_addr)==RC_OK)
    {
        memmove(&header, (char *)boot_addr, sizeof(image_header_t));
        /* vaildate magic
         */
        if (header.ih_magic != IH_MAGIC)
        {
            debug("%s:Bad runtime file(image header magic byte error)!, file_name=%s\n", __FUNCTION__, file_name);
            return RC_ERROR;
        }

        /* validate image header crc
         */
        checksum = ntohl(header.ih_hcrc);
        header.ih_hcrc = 0;
        if(crc32(0, (uchar*)&header, sizeof(image_header_t)) != checksum)
        {
            debug("%s:Bad image header checksum, file_name=%s\n", __FUNCTION__, file_name);
            return RC_ERROR;
        }

        /* validate image data crc
         */
        if(crc32(0, (uchar*)boot_addr + sizeof(image_header_t), file_size-sizeof(image_header_t))!=ntohl(header.ih_dcrc))
        {
            debug("%s:Bad image data checksum, file_name=%s\n", __FUNCTION__, file_name);
            return RC_ERROR;
        }
        
    }
    else
        return RC_ERROR;

    return RC_OK;
}

/* ------------------------------------------------------------------------
 * FUNCTION NAME: AMS_PART_delete_runtime_file
 * PURPOSE:	delete runtime file in flash device.
 * INPUT:	file name
 * OUTPUT:	None
 * RETURN:	
 *              RC_OK    = Success
 *              RC_ERROR = Fail
 * NOTES:
 * ------------------------------------------------------------------------
 */
int AMS_PART_delete_runtime_file(char *file_name)
{
    AMS_PARTITION_DEVICE_TYPE_T device_type;
    AMS_PARTITION_TYPE_T type = TYPE_RUNTIME;
    unsigned long start_addr = 0;
    unsigned long size = 0;
    unsigned long partition_index = 0;
    unsigned long ret;
    char *argv[3];

    if(AMS_PART_get_info_by_filename((char *)file_name, &start_addr, &size, &partition_index,&device_type) != RC_OK)
    {
        debug("%s:Cannot find file - %s", __FUNCTION__, file_name);
        return RC_ERROR;
    }

    debug("%s:[partition_index]%d, [start_addr]0x%X, [size]%ld\n", __FUNCTION__, partition_index, start_addr, size);

    if(AMS_PART_check_active(partition_index, type) == RC_OK)
    {
        ret = AMS_CMD_PART_set_startup_flag(partition_index, AMS_PARTITION_INACTIVE);
        
        if (ret == RC_PART_NUMBER_EXCEED || ret == RC_PART_DO_NOT_EXIST)
        {
            debug("%s:Invalid partition index: %d", __FUNCTION__, partition_index);
            return RC_ERROR;
        }
    }

    argv[0] = &file_name[0];
    argv[1] = &file_name[0];
    if(do_delete(NULL, 0, 2, argv)==0)
        return RC_OK;

    debug("%s:delete file error", __FUNCTION__);
    return RC_ERROR;
}

/* ------------------------------------------------------------------------
 * FUNCTION NAME: AMS_PART_prepare_runtime_image
 * PURPOSE: Check file partition table and select a bootable runtime image.
 *          If a bootable runtime image is not available, this function will
 *          try to recover a usable runtime image in flash through the
 *          validation of checksum.
 * INPUT:   None
 * OUTPUT:  boot_runtime_filename -- the filename of the runtime that is
 *                                   being used to boot up.
 *                                   The size of boot_runtime_filename must be
 *                                   greater or equal to PART_FILE_NAMELEN.
 * RETURN:  RC_OK    = Success
 *          RC_ERROR = Fail (No runtime image availabe for boot)
 * NOTES: 1.If RC_OK is returned, the bootable runtime image will be loaded
 *          at LA_DB_ADDR.
 *        2.Invalid runtime will be deleted from file mapping table.
 *        3.The startup runtime could be changed if the original startup
 *          runtime image is invalid.
 * ------------------------------------------------------------------------
 */
int AMS_PART_prepare_runtime_image(char* boot_runtime_filename)
{
    int img_runtime = FALSE;    /* The default runtime image is ready or not */
    unsigned long length;
    char          *local_args[2];

    /************************************************************************
     * Check default image file and get file name from file system.
     */
    if (AMS_PART_get_startup_filename(TYPE_RUNTIME, (unsigned char*)boot_runtime_filename, &length) == RC_OK)
    {
        if (AMS_PART_validate_and_load_boot_runtime()==RC_OK)
        {
            img_runtime = TRUE;
            debug("The default runtime image : %s, file size = %ld\r\n", boot_runtime_filename, length);
        }
        else
        {
            /* delete invalid runtime image
             */
            printf("Delete invalid runtime: %s\r\n", boot_runtime_filename);
            AMS_PART_delete_runtime_file((char*)boot_runtime_filename);

            /* default startup runtime image is invalid, try to use another
             * non-startup runtime image
             */
            printf("The default runtime image is invalid( %s, file size = %ld)\r\n", boot_runtime_filename, length);
            if (AMS_PART_get_non_startup_filename(TYPE_RUNTIME, (unsigned char*)boot_runtime_filename, &length) == RC_OK)
            {
                local_args[0] = (char *)boot_runtime_filename;
                local_args[1] = (char *)boot_runtime_filename;
                do_startup_file(NULL, 0, 2,local_args);
                /* validate the new startup runtime image
                 */
                if (AMS_PART_validate_and_load_boot_runtime()==RC_OK)
                {
                    img_runtime = TRUE;
                    printf("Set startup runtime image : %s, file size = %ld\r\n", boot_runtime_filename, length);
                }
                else
                {
                    printf("Non startup runtime image is invalid either. filename:%s, file size = %ld\r\n", boot_runtime_filename, length);
                    AMS_PART_delete_runtime_file((char*)boot_runtime_filename);
                }
            }
            else
            {
                printf("Non startup runtime image doesn not exist.\r\n");
                /* try our best to recover the possible usable runtimage on flash
                 */
                if (AMS_PART_recover_runtime_to_default()==RC_OK)
                {
                    if (AMS_PART_get_non_startup_filename(TYPE_RUNTIME, (unsigned char*)boot_runtime_filename, &length) == RC_OK)
                    {
                        img_runtime = TRUE;
                        printf("Runtime image is recovered successfully. (startup runtime to %s)\r\n", boot_runtime_filename);
                        local_args[0] = (char *)boot_runtime_filename;
                        local_args[1] = (char *)boot_runtime_filename;
                        do_startup_file(NULL, 0, 2,local_args); // set this file as startup type.
                        printf("The recovered runtime image : %s, file size = %ld\r\n", boot_runtime_filename, length);
                    }
                }
                else
                {
                    printf("Recover runtime image failed\r\n");
                }
                
            }
        }
    }
    else if (AMS_PART_get_non_startup_filename(TYPE_RUNTIME, (unsigned char*)boot_runtime_filename, &length) == RC_OK)
    {
        /* read the non-startup runtime image file from file system and start it.
         */
        printf("No startup runtime is set. Set startup runtime to %s\r\n", boot_runtime_filename);
        local_args[0] = (char *)boot_runtime_filename;
        local_args[1] = (char *)boot_runtime_filename;
        do_startup_file(NULL, 0, 2,local_args); // set this file as startup type.
        if (AMS_PART_validate_and_load_boot_runtime()==RC_OK)
        {
            img_runtime = TRUE;
            debug("The default runtime image : %s, file size = %ld\r\n", boot_runtime_filename, length);
        }
        else
        {
            printf("Delete invalid runtime image %s .\r\n", boot_runtime_filename);
            AMS_PART_delete_runtime_file((char*)boot_runtime_filename);

            /* try to get another non-startup runtime image file from file system
             */
            if(AMS_PART_get_non_startup_filename(TYPE_RUNTIME, (unsigned char*)boot_runtime_filename, &length) == RC_OK)
            {
                /* try to set as start-up and boot
                 */
                local_args[0] = (char *)boot_runtime_filename;
                local_args[1] = (char *)boot_runtime_filename;
                do_startup_file(NULL, 0, 2,local_args); // set this file as startup type.
                if (AMS_PART_validate_and_load_boot_runtime()==RC_OK)
                {
                    img_runtime = TRUE;
                    printf("The default runtime image : %s, file size = %ld\r\n", boot_runtime_filename, length);
                }
                else
                {
                    printf("Delete invalid runtime image %s .\r\n", boot_runtime_filename);
                    AMS_PART_delete_runtime_file((char*)boot_runtime_filename);
                }
            }
            else
            {
                /* try to recover one runtime
                 */
                debug("Try to recover a bootable runtime\r\n");
                if (AMS_PART_recover_runtime_to_default()==RC_OK)
                {
                    if (AMS_PART_get_non_startup_filename(TYPE_RUNTIME, (unsigned char*)boot_runtime_filename, &length) == RC_OK)
                    {
                        img_runtime = TRUE;
                        printf("runtime image is recovered sucessfully and will be set as startup runtime.(filename:%s)\r\n", boot_runtime_filename);
                        local_args[0] = (char *)boot_runtime_filename;
                        local_args[1] = (char *)boot_runtime_filename;
                        do_startup_file(NULL, 0, 2,local_args); // set this file as startup type.
                        debug("The default runtime image : %s, file size = %ld\r\n", boot_runtime_filename, length);
                    }
                }
            }
        }
    }
    else
    {
        /* No runtime file in file system, try to recover runtime.
         */
        printf("No runtime file in file system, try to recover.\r\n");
        if (AMS_PART_recover_runtime_to_default()==RC_OK)
        {
            if (AMS_PART_get_non_startup_filename(TYPE_RUNTIME, (unsigned char*)boot_runtime_filename, &length) == RC_OK)
            {
                img_runtime = TRUE;
                printf("runtime image is recovered sucessfully and will be set as startup runtime.(filename:%s)\r\n", boot_runtime_filename);
                local_args[0] = (char *)boot_runtime_filename;
                local_args[1] = (char *)boot_runtime_filename;
                do_startup_file(NULL, 0, 2,local_args); // set this file as startup type.
                debug("The default runtime image : %s, file size = %ld\r\n", boot_runtime_filename, length);
            }
        }
    }

    return (img_runtime==TRUE)? RC_OK : RC_ERROR;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_init_BPB
 * ------------------------------------------------------------------------
 * PURPOSE  :   Initialize boot parameter block (BPB).
 * INPUT    :   bpb_ptr = the base address of BPB data structure
 * OUTPUT   :   None
 * RETURN   :   
 *              RC_OK = partition table is correct.
 *              <others> = invalid partition table.
 * NOTES    :   None
 * ------------------------------------------------------------------------
 */
static int AMS_PART_init_BPB(AMS_BPB_T *bpb_ptr)
{
    if (bpb_ptr == NULL){
        printf("%s: invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }
	
    bpb_ptr->signature = AMS_BPB_SIGNATURE;
    bpb_ptr->length = sizeof(AMS_BPB_T) - sizeof(unsigned long);  //the size of AMS_BPB_T except for checksum field.

    memset((char *)&bpb_ptr->pat_table, 0, bpb_ptr->length);
    
    return AMS_PART_set_bpb_checksum(bpb_ptr);
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_set_bpb_checksum
 * ------------------------------------------------------------------------
 * PURPOSE  :   Find snd set the checksum in BPB.
 * INPUT    :   bpb_ptr = the base address of BPB data structure
 * OUTPUT   :   None
 * RETURN   :   
 *              RC_OK = partition table is correct.
 *              <others> = invalid partition table.
 * NOTES    :   None
 * ------------------------------------------------------------------------
 */
static int AMS_PART_set_bpb_checksum(AMS_BPB_T *bpb_ptr)
{
    if (bpb_ptr == NULL){
        debug("%s: invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    if ( bpb_ptr->signature != AMS_BPB_SIGNATURE){
        debug("%s: invalid data structure. Signature = 0x%lX\n", __FUNCTION__, bpb_ptr->signature);
        return RC_PARAMENT_ERROR;
    }

    bpb_ptr->checksum = CheckSum((unsigned long)bpb_ptr, bpb_ptr->length);
    
    return RC_OK;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_write_BackupBPB
 * ------------------------------------------------------------------------
 * PURPOSE  :   Write the backup BPB into flash device.
 * INPUT    :   None
 * OUTPUT   :   None
 * RETURN   :   
 *              RC_OK = partition table is correct.
 *              <others> = invalid partition table.
 * NOTES    :   None
 * ------------------------------------------------------------------------
 */
static int AMS_PART_write_BackupBPB(void)
{
    debug("%s: base=0x%08X, size=0x%08X\n", __FUNCTION__, PARTITION_BPB_BASE_ADDR, CFG_FLASH_SECTOR_SIZE);

    if(AMS_PART_set_bpb_checksum(m_bpb_backup) == RC_OK){            
#if (PARTITION_BPB_BASE_ADDR==0xffffffff)
        debug("[Debug] %s: No BPB Support, so No Backup!\n", __FUNCTION__);
#else
        AMS_PART_write_flash(PARTITION_BPB_BASE_ADDR, (unsigned char *)m_bpb_backup, 
            CFG_FLASH_SECTOR_SIZE, sizeof(AMS_BPB_T),1);
#endif
        return RC_OK;
    }

    debug("[Debug] %s: BPB error !\n", __FUNCTION__);    
    return RC_ERROR;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_set_primary_part_table
 * ------------------------------------------------------------------------
 * PURPOSE  :   Update the primary partition table in flash device.
 * INPUT    :   
 *              src = the base address of source partition table
 * OUTPUT   :   None
 * RETURN   :   
 *              RC_OK = partition table is correct.
 *              <others> = invalid partition table.
 * NOTES    :   None
 * ------------------------------------------------------------------------
 */
static int AMS_PART_set_primary_part_table(AMS_PARTITION_TABLE_T *src)
{
    if (src == NULL){
        debug("%s: invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }
	
    memcpy(&m_bpb_primary->pat_table, src, sizeof(AMS_PARTITION_TABLE_T));

    AMS_PART_set_partition_checksum(&m_bpb_primary->pat_table);
    AMS_PART_set_bpb_checksum(m_bpb_primary);

    debug("%s: base=0x%08X, size=0x%08X\n", __FUNCTION__, PARTITION_MANAGEMENT_BASE_ADDR, CFG_FLASH_SECTOR_SIZE);
    AMS_PART_write_flash(PARTITION_MANAGEMENT_BASE_ADDR, (unsigned char *)&m_bpb_primary->pat_table, 
        CFG_FLASH_SECTOR_SIZE, sizeof(AMS_PARTITION_TABLE_T), 1);
    
    return RC_OK;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_set_primary_filemap_table
 * ------------------------------------------------------------------------
 * PURPOSE  :   Update the primary file mapping table in flash device.
 * INPUT    :   dest = the base address of target file mapping table
 *              src = the base address of source file mapping table
 * OUTPUT   :   None
 * RETURN   :   
 *              RC_OK = partition table is correct.
 *              <others> = invalid partition table.
 * NOTES    :   None
 * ------------------------------------------------------------------------
 */
static int AMS_PART_set_primary_filemap_table(AMS_FILEMAPPING_MGR_T *src)
{
    unsigned long index, chksum;

    if (src == NULL){
        debug("%s: invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }
	
	
    memcpy(&m_bpb_primary->file_map, src, sizeof(AMS_FILEMAPPING_MGR_T));
    
    for (index = 0; index < MAX_PARTITION_NUMBER; index++){
        AMS_PART_find_filemap_element_checksum(&m_bpb_primary->file_map.filemapping[index], &chksum);
        m_bpb_primary->file_map.filemapping[index].part_checksum = chksum;
    }
    
	AMS_PART_write_filemapping2flash(&m_bpb_primary->file_map);
    AMS_PART_set_bpb_checksum(m_bpb_primary);
    
	return RC_OK;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_set_primary_hwinfo
 * ------------------------------------------------------------------------
 * PURPOSE  :   Update the primary H/W info in flash device.
 * INPUT    :   dest = the base address of target H/W infomation table
 *              src = the base address of source H/W infomation table
 * OUTPUT   :   None
 * RETURN   :   
 *              RC_OK = partition table is correct.
 *              <others> = invalid partition table.
 * NOTES    :   None
 * ------------------------------------------------------------------------
 */
static int AMS_PART_set_primary_hwinfo(FS_HW_Info_T *src)
{
    if (src == NULL){
        debug("%s: invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }
	
    
    memcpy(&m_bpb_primary->hw_info, src, sizeof(FS_HW_Info_T));

    AMS_PART_set_hwinfo_checksum(&m_bpb_primary->hw_info);
    AMS_PART_set_bpb_checksum(m_bpb_primary);

    if(AMS_PART_write_to_device(TYPE_HWINFO, PARTITION_HWINFO_FILENAME, (unsigned long)&m_bpb_primary->hw_info, sizeof(FS_HW_Info_T), PARTITION_HWINFO_INDEX) != RC_OK){
        printf("[Error] %s: Fail to save hardware information to flash\n", __FUNCTION__);
        return RC_ERROR;
    }
    
	return RC_OK;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_check_part_table
 * ------------------------------------------------------------------------
 * PURPOSE  :   Check partition table.
 * INPUT    :   None
 * OUTPUT   :   None
 * RETURN   :   
 *              RC_OK = partition table is correct.
 *              <others> = invalid partition table.
 * NOTES    :   None
 * ------------------------------------------------------------------------
 */
static int AMS_PART_check_part_table(AMS_PARTITION_TABLE_T *part_tbl)
{
    unsigned long chksum;

    if (part_tbl == NULL){
        debug("%s: invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }
	    
    
    if(part_tbl->magic != PARTITION_TABLE_MAGIC){
        debug("Invalid magic word in partition table: 0x%lX\n", part_tbl->magic);
        return RC_ERROR;
    }

    
    if(part_tbl->version != CFG_AMS_PARTITION_VERSION){
        debug("Invalid version word in partition table: 0x%lX\n", part_tbl->version);
        return RC_ERROR;
    }
        
        
    chksum = CheckSum((unsigned long)&part_tbl->magic, 16);
    if(part_tbl->header_checksum != chksum){
        debug("Checksum mismatch !\npart_tbl->header_checksum: 0x%lX (0x%lX)\n", part_tbl->header_checksum, chksum);
        return RC_ERROR;
    }

    return RC_OK;

}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_check_filemap_table
 * ------------------------------------------------------------------------
 * PURPOSE  :   Check file mapping table.
 * INPUT    :   None
 * OUTPUT   :   None
 * RETURN   :   
 *              RC_OK = file mapping table is correct.
 *              <others> = invalid file mapping table.
 * NOTES    :   None
 * ------------------------------------------------------------------------
 */
static int AMS_PART_check_filemap_table(AMS_FILEMAPPING_MGR_T *fmt)
{
    int i;
    unsigned long chksum;
    
    if (fmt == NULL){
        debug("%s: invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    for(i=0; i < MAX_PARTITION_NUMBER; i++){
        if(fmt->filemapping[i].file_type == TYPE_NOT_USED)
            continue;
        
        chksum = CheckSum((unsigned long)&fmt->filemapping[i].file_name[0], 
             sizeof(AMS_FILEMAPPING_T) - sizeof(unsigned long));
      
	    if(fmt->filemapping[i].part_checksum != chksum){
            return RC_ERROR;
        }
    }
    
    return RC_OK;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_check_hwinfo_table
 * ------------------------------------------------------------------------
 * PURPOSE  :   Check file mapping table.
 * INPUT    :   None
 * OUTPUT   :   None
 * RETURN   :   
 *              RC_OK = file mapping table is correct.
 *              <others> = invalid file mapping table.
 * NOTES    :   None
 * ------------------------------------------------------------------------
 */
static int AMS_PART_check_hwinfo_table(FS_HW_Info_T *hit)
{
    unsigned short chksum;
    
    if (hit == NULL){
        debug("%s: invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    chksum = (unsigned short)CheckSum((unsigned long)hit, sizeof(FS_HW_Info_T) - sizeof(unsigned short));    

    //debug("[Debug] %s: chksum = 0x%lX, hit->check_sum = 0x%lX\n", __FUNCTION__, chksum, hit->check_sum);
    if((chksum != hit->check_sum) || (chksum == 0xffff) )
        return RC_ERROR;
    
    return RC_OK;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_get_partbl_field_by_type
 * ------------------------------------------------------------------------
 * PURPOSE  :   Get data field of partition table with specific type.
 * INPUT    :   type = partition type
 * OUTPUT   :   region = the data field of partition table. 
 * RETURN   :   
 *              RC_OK = Success
 *              <others> = Fail
 * NOTES    :   None
 * ------------------------------------------------------------------------
 */
static int AMS_PART_get_partbl_field_by_type(AMS_PARTITION_TABLE_T *part_tbl, AMS_PARTITION_TYPE_T type, FlashRegion_T *region)
{
    int i;
    
    if ((region == NULL) || (part_tbl == NULL)){
        debug("%s: invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    for (i = 0; i < MAX_PARTITION_NUMBER; i++){
        if (part_tbl->flash_region[i].region_type == type){
            memcpy((unsigned char *)region, (unsigned char *)&part_tbl->flash_region[i], sizeof(FlashRegion_T));

            return RC_OK;
        }
    }

    return RC_PART_DO_NOT_EXIST;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_verify_primary
 * ------------------------------------------------------------------------
 * PURPOSE  :   Verify the primary partition table, file mapping table and H/W info.
 * INPUT    :   part_tbl = the base address of partition table.
 * OUTPUT   :   None. 
 * RETURN   :   
 *              RC_OK = Success
 *              <others> = Fail
 * NOTES    :  
 *              Set file mapping table to invalid if partition table is invalid.
 * ------------------------------------------------------------------------
 */
static int AMS_PART_verify_primary(AMS_PARTITION_TABLE_T *part_tbl)
{
    FlashRegion_T   region;
    AMS_FILEMAPPING_MGR_T *p_filemapping_table;
    FS_HW_Info_T    *p_hwinfo;
    int flag;

    if (part_tbl == NULL){
        debug("%s: invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    flag = BPB_FLAG_PRIMARY;   //indicate that it is primary data. 
    
    /* step 1. verify partition table first.
     */
    if (AMS_PART_check_part_table(part_tbl) != RC_OK){        
        flag |= (BPB_FLAG_PT | BPB_FLAG_FMT);
        return flag;
    }

    
    /* step 2. verify others 
     */
    //2-1. verify filemapping table only if partition table is valid.
    if(AMS_PART_get_partbl_field_by_type(part_tbl, TYPE_FILEMAPPING_TABLE, &region) == RC_OK){
        p_filemapping_table = (AMS_FILEMAPPING_MGR_T *)region.region_base;
       
        if (AMS_PART_check_filemap_table(p_filemapping_table) != RC_OK){
            flag |= BPB_FLAG_FMT;
        }
    
    }else{
        debug("[Debug] %s: failed to get filemapping from PT.\n", __FUNCTION__);
        flag |= BPB_FLAG_FMT;
    }

    
    //2-2. verify H/W infomation.
    if(AMS_PART_get_partbl_field_by_type(part_tbl, TYPE_HWINFO, &region) == RC_OK){
        p_hwinfo = (FS_HW_Info_T *)region.region_base;

        if (AMS_PART_check_hwinfo_table(p_hwinfo) != RC_OK){
    	    flag |= BPB_FLAG_HIT;
        }
                
    }else{
        debug("[Debug] %s: failed to get H/W info from PT.\n", __FUNCTION__);
        flag |= BPB_FLAG_HIT;
    }

    return flag;

}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_verify_backup
 * ------------------------------------------------------------------------
 * PURPOSE  :   Verify the backup partition table, file mapping table and H/W info.
 * INPUT    :   part_tbl = the base address of backup BPB.
 * OUTPUT   :   None. 
 * RETURN   :   
 *              RC_OK = Success
 *              <others> = Fail
 * NOTES    :
 *              Set file mapping table to invalid if partition table is invalid.
 *              Set all partitions to invalid if checksum is invalid. 
 * ------------------------------------------------------------------------
 */
static int AMS_PART_verify_backup(AMS_BPB_T *bpb)
{
    int flag;

    if (bpb == NULL){
        debug("%s: invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    flag = BPB_FLAG_BAK;   //indicate that it is backup data. 

    /* step 1. verify backup boot parameter block first.
     */
    // boot parameter block
    if(bpb->signature != AMS_BPB_SIGNATURE){
        flag |= (BPB_FLAG_PT | BPB_FLAG_FMT | BPB_FLAG_HIT);
        return flag;
    }

    if(bpb->checksum != CheckSum((unsigned long)bpb, bpb->length)){
        flag |= (BPB_FLAG_PT | BPB_FLAG_FMT | BPB_FLAG_HIT);
        return flag;
    }

    /* step 2. verify partition table.
     */    
    // partition table   
    if (AMS_PART_check_part_table(&bpb->pat_table) != RC_OK){
    	flag |= BPB_FLAG_PT;
    }
    

    /* step 3. verify others 
     */
    // 3-1. verify filemapping table only if partition table is valid.
    if( (flag & BPB_FLAG_PT) != 0){
        //Set file mapping table to invalid if partition table is invalid.
        flag |= BPB_FLAG_FMT;
    }else{
        if (AMS_PART_check_filemap_table(&bpb->file_map) != RC_OK){
            flag |= BPB_FLAG_FMT;
        }
    }
    
    
    // 3-2. verify H/W infomation.
    if (AMS_PART_check_hwinfo_table(&bpb->hw_info) != RC_OK){
        flag |= BPB_FLAG_HIT;
    }

    return flag;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_recover_part_table
 * ------------------------------------------------------------------------
 * PURPOSE  :   To recover the partition table from the valid data source.
 *              The valid data souce is the primary data, the backup data and default value.
 * INPUT    :   flag_prim = the status of primary BPB.
 *              flag_bak = the status of backup BPB.
 * OUTPUT   :   None. 
 * RETURN   :   
 *              RC_OK = Success
 *              <others> = Fail
 * NOTES    :   None
 * ------------------------------------------------------------------------
 */
static int AMS_PART_recover_part_table(int flag_prim, int flag_bak)
{
    int state;
    
    state = (flag_prim & BPB_FLAG_PT) | ((flag_bak & BPB_FLAG_PT)<<1);

    
    switch(state){
        case 1:     /* primary: NG, backup: OK */  
            debug("\n[Error] Invalid primary partition table.\n");
            debug("Recover from the backup.\n");
            AMS_PART_set_primary_part_table(&m_bpb_backup->pat_table);
            break;
                        
        case 2:     /* primary: OK, backup: NG */
            debug("\n[Error] Invalid backup partition table.\n");
            debug("Recover from the primary.\n");
            memcpy(&m_bpb_backup->pat_table, &m_bpb_primary->pat_table, sizeof(AMS_PARTITION_TABLE_T));
            AMS_PART_set_bpb_checksum(m_bpb_backup);
            m_bpb_backup_dirty = 1;            
            break;
            
        case 3:     /* both are NG */
            debug("\n[Error] No valid partition table.\n");
            debug("Using the default value.\n");
            //primary + backup
            AMS_PART_set_primary_part_table(&m_bpb_default->pat_table);
            memcpy(&m_bpb_backup->pat_table, &m_bpb_default->pat_table, sizeof(AMS_PARTITION_TABLE_T));
            AMS_PART_set_bpb_checksum(m_bpb_backup);
            m_bpb_backup_dirty = 1;

    }

    return state;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_recover_filemap_table
 * ------------------------------------------------------------------------
 * PURPOSE  :   To recover the file mapping table from the valid data source.
 *              The valid data souce is the primary data, the backup data and default value.
 * INPUT    :   flag_prim = the status of primary BPB.
 *              flag_bak = the status of backup BPB.
 * OUTPUT   :   None. 
 * RETURN   :   
 *              RC_OK = Success
 *              <others> = Fail
 * NOTES    :   If the status of primary and backup BPB are valid, 
 *              then it uses the primary BPB.
 * ------------------------------------------------------------------------
 */
static int AMS_PART_recover_filemap_table(int flag_prim, int flag_bak)
{
    int state;
    AMS_FILEMAPPING_MGR_T *current_fmt;
    
    state = ((flag_prim & BPB_FLAG_FMT) | ((flag_bak & BPB_FLAG_FMT)<<1)) >> 1;
    
    switch(state){
        case 1:     /* primary: NG, backup: OK */  
            debug("\n[Error] Invalid primary file mapping table.\n");
            debug("Recover from the backup.\n");
            AMS_PART_set_primary_filemap_table(&m_bpb_backup->file_map);            
            break;
        
        case 2:     /* primary: OK, backup: NG */
            debug("\n[Error] Invalid backup file mapping table.\n");
            debug("Recover from the primary.\n");
            AMS_PART_get_filemapping_mgr_ptr(&current_fmt);
            memcpy(&m_bpb_backup->file_map, current_fmt, sizeof(AMS_FILEMAPPING_MGR_T));
            AMS_PART_set_bpb_checksum(m_bpb_backup);
            m_bpb_backup_dirty = 1;            
            break;
                        
        case 3:     /* primary + backup are NG */
            debug("\n[Error] No valid file mapping table.\n");
            debug("Using the default value.\n");
            AMS_PART_set_primary_filemap_table(&m_bpb_default->file_map);
            memcpy(&m_bpb_backup->file_map, &m_bpb_default->file_map, sizeof(AMS_FILEMAPPING_MGR_T));
            AMS_PART_set_bpb_checksum(m_bpb_backup);
            m_bpb_backup_dirty = 1;
            break;
            
        default:
            //get one copy from flash to primary BPB (in DRAM).
            AMS_PART_get_filemapping_mgr_ptr(&current_fmt);
            memcpy((unsigned char *)&m_bpb_primary->file_map, current_fmt, sizeof(AMS_FILEMAPPING_MGR_T));                
            AMS_PART_set_bpb_checksum(m_bpb_primary);
    }

    return state;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_recover_hwinfo
 * ------------------------------------------------------------------------
 * PURPOSE  :   To recover the H/W information table from the valid data source.
 *              The valid data souce is the primary data, the backup data and default value.
 * INPUT    :   flag_prim = the status of primary BPB.
 *              flag_bak = the status of backup BPB.
 * OUTPUT   :   None. 
 * RETURN   :   
 *              RC_OK = Success
 *              <others> = Fail
 * NOTES    :   If the status of primary and backup BPB are valid, 
 *              then it uses the primary BPB.
 * ------------------------------------------------------------------------
 */
static int AMS_PART_recover_hwinfo(int flag_prim, int flag_bak)
{
    int state;
    FS_HW_Info_T hit;
    
    state = ((flag_prim & BPB_FLAG_HIT) | ((flag_bak & BPB_FLAG_HIT)<<1)) >> 2;
    
    switch(state){
        case 1:     /* primary: NG, backup: OK */
            debug("\n[Error] Invalid primary hardware information.\n");
            debug("Recover from the backup.\n");
            AMS_PART_set_primary_hwinfo(&m_bpb_backup->hw_info);                       
            break;
        
        case 2:     /* primary: OK, backup: NG */ 
            debug("\n[Error] Invalid backup hardware information.\n");
            debug("Recover from the primary.\n");
            AMS_PART_get_hwinfo(&hit);
            memcpy(&m_bpb_backup->hw_info, &hit, sizeof(FS_HW_Info_T));
            AMS_PART_set_bpb_checksum(m_bpb_backup);
            m_bpb_backup_dirty = 1;            
            break;
                        
        case 3:     /* primary + backup are NG */
            debug("\n[Error] No valid hardware information.\n");
            debug("Using the default value.\n");
            AMS_PART_set_primary_hwinfo(&m_bpb_default->hw_info);
            memcpy(&m_bpb_backup->hw_info, &m_bpb_default->hw_info, sizeof(FS_HW_Info_T));            
            AMS_PART_set_bpb_checksum(m_bpb_backup);
            m_bpb_backup_dirty = 1;
            break;
            
        default:
            //get one copy from flash to primary BPB (in DRAM).
            AMS_PART_get_hwinfo(&hit);
            memcpy(&m_bpb_primary->hw_info, &hit, sizeof(FS_HW_Info_T));                
            AMS_PART_set_bpb_checksum(m_bpb_primary);
    }

    return state;
}

U_BOOT_CMD(
	bpb, CFG_MAXARGS, 1,	do_BPB,
	"",
	"   -The boot parameter block(BPB) utility\n"
);
static int do_BPB(cmd_tbl_t *cmdtp, int flag, int argc, char *argv[])
{
    printf("\nprimary BPB\n");
    printf("\tmemory address: 0x%lX\n", m_bpb_primary);    
    printf("\t[checksum] 0x%lX [size] %ld\n", m_bpb_primary->checksum, m_bpb_primary->length);
    
    printf("\nbackup BPB:\n");
    printf("\tmemory address: 0x%lX\n", m_bpb_backup);  
    printf("\tflash address: 0x%lX\n", PARTITION_BPB_BASE_ADDR); 
    printf("\t[checksum] 0x%lX [size] %ld\n", m_bpb_backup->checksum, m_bpb_backup->length);

    printf("\ndefault BPB:\n");
    printf("\taddress: 0x%lX\n", m_bpb_default);    
    printf("\t[checksum] 0x%lX [size] %ld\n", m_bpb_default->checksum, m_bpb_default->length);
    
    printf("\nsizeof(AMS_BPB_T) = %d\n", sizeof(AMS_BPB_T));        
    printf("sizeof(AMS_PARTITION_TABLE_T) = %d\n", sizeof(AMS_PARTITION_TABLE_T));        
    printf("sizeof(AMS_FILEMAPPING_MGR_T) = %d\n", sizeof(AMS_FILEMAPPING_MGR_T));        
    printf("sizeof(FS_HW_Info_T) = %d\n", sizeof(FS_HW_Info_T));        
    
    return 0;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_recover
 * ------------------------------------------------------------------------
 * PURPOSE  :   To recover the primary or backup or both from the valid data source.
 *              The valid data souce is one of the primary data, the backup data and default value.
 * INPUT    :   flag_prim = the status of primary BPB.
 *              flag_bak = the status of backup BPB.
 * OUTPUT   :   None. 
 * RETURN   :   
 *              RC_OK = Success
 *              <others> = Fail
 * NOTES    :   If the status of primary and backup BPB are valid, 
 *              then it uses the primary BPB.
 * ------------------------------------------------------------------------
 */
static int AMS_PART_recover_bpb(int flag_prim, int flag_bak)
{
    if((flag_bak & BPB_FLAG_MASK_DATA) == (BPB_FLAG_PT | BPB_FLAG_FMT | BPB_FLAG_HIT)){
        AMS_PART_init_BPB(m_bpb_backup);
    }
    

    // 1st stage: recover partition table.    
    AMS_PART_recover_part_table(flag_prim, flag_bak);
    if(flag_prim & BPB_FLAG_PT){
        //Recover file mapping table if partition table is invalid.
        AMS_PART_recover_filemap_table(flag_prim, flag_bak);
        flag_prim = AMS_PART_verify_primary(&m_bpb_primary->pat_table);
    }
    if(flag_bak & BPB_FLAG_PT){
        flag_bak &= ~BPB_FLAG_PT;
    }

    // 2nd stage: recover others.
    debug("[Debug] Table Status after recover partition table: primary = 0x%X, backup = 0x%X\n", flag_prim, flag_bak);
    AMS_PART_recover_filemap_table(flag_prim, flag_bak);
    AMS_PART_recover_hwinfo(flag_prim, flag_bak);

    if(m_bpb_backup_dirty != 0){
        AMS_PART_write_BackupBPB();
    }

    return RC_OK;

}


/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_init
 * ------------------------------------------------------------------------
 * PURPOSE  :   Initialize the flash management system.
 * INPUT    :   
 *              def_pt      = default partition table
 *              def_hit     = default H/W information
 *              rec_runtime = recover runtime file I/F.
 * OUTPUT   :   None
 * RETURN   :   
 *              RC_OK   = Success
 *              RC_ERROR  = Fail
 *
 * NOTES    :   NONE
 * ------------------------------------------------------------------------
 */
int AMS_PART_init(AMS_PARTITION_TABLE_T *def_pt, FS_HW_Info_T *def_hit)
{
    int sts_primary, sts_bak, i;
    int sts_primary_new, sts_bak_new;
    char *env;
    char buf[200];

    if((m_ams_flag & AMS_FLAG_INIT) > 0){
        debug("%s(): AMS has been initial !\n", __FUNCTION__);
        return RC_OK;
    }
        
    sts_primary = sts_bak = 0;
    sts_primary_new = sts_bak_new = 0;

    if((def_pt == NULL) || (def_hit == NULL)){
        debug("[Error] The default partition table or H/W information is not available.\n");
        debug("Cannot initialize boot parameter block.\n");
        return RC_ERROR;
    }

    m_bpb_primary = (AMS_BPB_T *)malloc(sizeof(AMS_BPB_T));        
    m_bpb_backup = (AMS_BPB_T *)malloc(sizeof(AMS_BPB_T));
    m_bpb_default = (AMS_BPB_T *)malloc(sizeof(AMS_BPB_T));
    if((m_bpb_primary == NULL) || (m_bpb_backup == NULL) || (m_bpb_default == NULL)){
        debug("[Error] Fail to allocate memory !\n");
        debug("Cannot initialize boot parameter block.\n");
        return RC_ERROR;
    }
    AMS_PART_init_BPB(m_bpb_primary);
    AMS_PART_init_BPB(m_bpb_default);

    /* Get data from flash memory.
     * Only the partition table of primary BPB is valid from flash device.
     * The backup BPB has whole data from flash device.
     */
    memcpy((unsigned char *)&m_bpb_primary->pat_table, (unsigned char *)PARTITION_MANAGEMENT_BASE_ADDR, sizeof(AMS_PARTITION_TABLE_T));    
    AMS_PART_set_bpb_checksum(m_bpb_primary);
#if (PARTITION_BPB_BASE_ADDR!=0xffffffff)
    memcpy((unsigned char *)m_bpb_backup, (unsigned char *)PARTITION_BPB_BASE_ADDR, sizeof(AMS_BPB_T));
#else
    memcpy((unsigned char *)m_bpb_backup, (unsigned char *)m_bpb_primary, sizeof(AMS_BPB_T));
#endif
    m_bpb_backup_dirty = 0;


    //set default value for default partition table , file map and H/W info.
    memcpy((unsigned char *)&m_bpb_default->pat_table, (unsigned char *)def_pt, sizeof(AMS_PARTITION_TABLE_T));
    memcpy((unsigned char *)&m_bpb_default->hw_info, (unsigned char *)def_hit, sizeof(FS_HW_Info_T));
    AMS_PART_set_hwinfo_checksum(&m_bpb_default->hw_info);
    AMS_PART_set_partition_checksum(&m_bpb_default->pat_table);
    for(i=0; i < MAX_PARTITION_NUMBER; i++){
        AMS_PART_init_filemapping(i, &m_bpb_default->file_map.filemapping[i]);
    }
    AMS_PART_set_bpb_checksum(m_bpb_default);
   
    m_ams_flag |= AMS_FLAG_INIT;    //AMS initial done.
    

    /*
     * To recover the primary or backup or both boot parameter record 
     * if error occurred.
     */    
    sts_primary = AMS_PART_verify_primary(&m_bpb_primary->pat_table);
    sts_bak = AMS_PART_verify_backup(m_bpb_backup);    
    sprintf(buf, "[primary] 0x%X [backup] 0x%X", sts_primary, sts_bak);    
    setenv("bstatus", buf);
    debug("[Debug] BPB status: %s\n", buf);

    if((env = getenv("recoverbpb")) != NULL){
        // do NOT recover boot parameter record if  "recoverbpb=off"
        if(strcmp(env, "off") == 0)     
            return RC_OK;
    }

    if( (sts_primary & BPB_FLAG_MASK_DATA) || (sts_bak & BPB_FLAG_MASK_DATA) ){ //if error happen
        AMS_PART_recover_bpb(sts_primary, sts_bak);

        //get data from flash memory.
        memcpy((unsigned char *)&m_bpb_primary->pat_table, (unsigned char *)PARTITION_MANAGEMENT_BASE_ADDR, sizeof(AMS_PARTITION_TABLE_T));    
        AMS_PART_set_bpb_checksum(m_bpb_primary);
#if (PARTITION_BPB_BASE_ADDR!=0xffffffff)
        memcpy((unsigned char *)m_bpb_backup, (unsigned char *)PARTITION_BPB_BASE_ADDR, sizeof(AMS_BPB_T));
#else
        memcpy((unsigned char *)m_bpb_backup, (unsigned char *)m_bpb_primary, sizeof(AMS_BPB_T));
#endif

        // Verify primary and backup data again
        sts_primary_new = AMS_PART_verify_primary(&m_bpb_primary->pat_table);
        sts_bak_new = AMS_PART_verify_backup(m_bpb_backup);

        sprintf(buf, "(after recovered) 0x%X, 0x%X", sts_primary_new, sts_bak_new);
        setenv("bstatus_new", buf);
        debug("[Debug] BPB status: %s\n", buf);
    }
    
    return RC_OK;
}

#endif   /* #CONFIG_AMS_PARTITION */

